/**
 * This class demonstrates how application programmers can utilize the
 * APIs in the following packages: javax.xml.parser, javax.xml.transform, 
 * javax.xml.transform.sax, javax.xml.transform.dom and javax.xml.transform.stream.
 *
 * Each method implemented in this class shows one way to perform an XML/XSLT transformation.
 */

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.sax.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;

import java.io.*;
import java.util.*;
import java.net.URL;
import java.net.MalformedURLException;

import org.xml.sax.*;
import org.xml.sax.ext.*;
import org.xml.sax.helpers.*;
import org.w3c.dom.*;

public class JAXPExamples
{
	public static void main(String argv[])
   	throws TransformerException, TransformerConfigurationException,
        IOException, SAXException,
        ParserConfigurationException, FileNotFoundException
  	{
    	try {
	 // use the jaxpone.xml, jaxpone.xsl, general.xml, age.xsl, jaxptwo.xsl
	 // and jaxpthree.xsl files for this demo.
         URL xmlURL = createURL("jaxpone.xml");
         String xmlID = xmlURL.toString();
         URL xslURL = createURL("jaxpone.xsl");
         String xslID = xslURL.toString();
         //
         System.out.println("--- basic ---");
         basic(xmlID, xslID);
         System.out.println();
         System.out.println("--- identity ---");
         identity(xmlID);
         //
         URL generalURL = createURL("general.xml");
         String generalID = generalURL.toString();
         URL ageURL = createURL("age.xsl");
         String ageID = ageURL.toString();
         System.out.println();
         System.out.println("--- namespaceURI ---");
         namespaceURI(generalID, ageID);
         //
         System.out.println();
         System.out.println("--- templatesHandler ---");
         templatesHandler(xmlID, xslID);
         System.out.println();
         System.out.println("--- contentHandler2contentHandler ---");
         contentHandler2contentHandler(xmlID, xslID);
         System.out.println();
         System.out.println("--- contentHandler2DOM ---");
         contentHandler2DOM(xmlID, xslID);
         System.out.println();
         System.out.println("--- reader ---");
         reader(xmlID, xslID);
         System.out.println();
         System.out.println("--- xmlFilter ---");
         xmlFilter(xmlID, xslID);
         //
         URL xslURLtwo = createURL("jaxptwo.xsl");
         String xslIDtwo = xslURLtwo.toString();
         URL xslURLthree = createURL("jaxpthree.xsl");
         String xslIDthree = xslURLthree.toString();
         System.out.println();
         System.out.println("--- xmlFilterChain ---");
         xmlFilterChain(xmlID, xslID, xslIDtwo, xslIDthree);
      } catch(Exception err) {
      	err.printStackTrace();
      }
   }
  
   /**
    * Perform a basic transformation.
    * @param xmlID - the source XML
    * @param xslID - the stylesheet 
    */
   public static void basic(String xmlID, String xslID)
      throws TransformerException, TransformerConfigurationException
   {
      // get a TransformerFactory object by calling static method newInstance() 
      // in TransformerFactory
      TransformerFactory tfactory = TransformerFactory.newInstance();
      // the Transformer object now includes the stylesheet infomation
      Transformer transformer = tfactory.newTransformer(new StreamSource(xslID));
      // generate a StreamSource object from the XML source
      StreamSource source = new StreamSource(xmlID);
      // based on the stylesheet (included in transformer) and XML source, 
      // we generate the output to standard System Output Device
      transformer.transform(source, new StreamResult(System.out));
   }

   /**
    * Perform an identity transformation, which is a transformation
    * that does not use a stylesheet. The input XML document is output.
    * @param xmlID - the source XML
    */
   public static void identity(String xmlID)
      throws TransformerException, TransformerConfigurationException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();
      Transformer transformer = tfactory.newTransformer();
      transformer.setOutputProperty(OutputKeys.METHOD, "html");
      transformer.setOutputProperty(OutputKeys.INDENT, "no");
      StreamSource source = new StreamSource(xmlID);
      // transform input into itself because stylesheet is not used
      transformer.transform(source, new StreamResult(System.out));
   }

   /**
    * Sets stylesheet parameters with non-null namespace through APIs.
    * @param xmlID - the source XML
    * @param xslID - the stylesheet 
    */
   public static void namespaceURI(String xmlID, String xslID)
      throws TransformerException, TransformerConfigurationException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();
      Transformer transformer
         = tfactory.newTransformer(new StreamSource(xslID));
      System.out.println("default: 99");
      transformer.transform( new StreamSource(xmlID),
         new StreamResult(System.out));
      transformer.setParameter("{http://www.oracle.com/ages}age", "20");
      System.out.println();
      System.out.println("should say: 20");
      transformer.transform( new StreamSource(xmlID),
         new StreamResult(System.out));
      transformer.setParameter("{http://www.oracle.com/ages}age", "40");
      transformer.setOutputProperty(OutputKeys.METHOD, "html");
      System.out.println();
      System.out.println("should say: 40");
      transformer.transform( new StreamSource(xmlID),
         new StreamResult(System.out));
   }

   /**
    * Performs a transformation with SAX. A TemplatesHandler object is used 
    * as the ContentHandler.
    * @param xmlID - the source XML
    * @param xslID - the stylesheet 
    */   
   public static void templatesHandler(String xmlID, String xslID)
      throws TransformerException, TransformerConfigurationException,
      IOException, SAXException,
      ParserConfigurationException, FileNotFoundException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();
      // test if we support SAXTransformerFactory
      if (tfactory.getFeature(SAXTransformerFactory.FEATURE))
      {
         SAXTransformerFactory stfactory = (SAXTransformerFactory) tfactory;
         TemplatesHandler handler = stfactory.newTemplatesHandler();
         handler.setSystemId(xslID);
         // set the SAX driver
         System.setProperty("org.xml.sax.driver", "oracle.xml.parser.v2.SAXParser");
         // create and config the parser
         XMLReader reader = XMLReaderFactory.createXMLReader();
         // when a TemplatesHandler is used as a ContentHandler
         // for parsing XSL instructions, it creates a Templates object
         reader.setContentHandler(handler);
         // parse the stylesheet
         reader.parse(xslID);
         // obtain the Templates object
         Templates templates = handler.getTemplates();
         // now the transformer has the compiled stylesheet (or templates) 
         // generated by SAX events.
         Transformer transformer = templates.newTransformer();
         transformer.transform(new StreamSource(xmlID), new StreamResult(System.out));
      }
   }


   /**
    * Performs a transformation with SAX by calling XMLReader.parse().
    * The output depends on how user implements ContentHandler
    * @param xmlID - the source XML
    * @param xslID - the stylesheet 
    */   
   public static void reader(String xmlID, String xslID)
      throws TransformerException, TransformerConfigurationException,
      SAXException, IOException, ParserConfigurationException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();
      SAXTransformerFactory stfactory = (SAXTransformerFactory)tfactory;
      StreamSource streamSource = new StreamSource(xslID);
      // create the parser
      XMLReader reader = stfactory.newXMLFilter(streamSource);
      // register a simply implemented ContentHandler class with the parser
      ContentHandler contentHandler = new oraContentHandler();
      reader.setContentHandler(contentHandler);
      InputSource is = new InputSource(xmlID);
      // parse and transform the XML document
      reader.parse(is);
   }

   /**
    * Performs a transformation with SAX by calling XMLFilter.parse().
    * The output depends on how the user implements ContentHandler.
    * @param xmlID - the source XML
    * @param xslID - the stylesheet 
    */ 
   public static void xmlFilter(String xmlID, String xslID)
      throws TransformerException, TransformerConfigurationException,
      SAXException, IOException, ParserConfigurationException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();
      XMLReader reader = null;
      try {
         javax.xml.parsers.SAXParserFactory factory =
	    javax.xml.parsers.SAXParserFactory.newInstance();
	 factory.setNamespaceAware(true);
         javax.xml.parsers.SAXParser jaxpParser =
	    factory.newSAXParser();
	 reader = jaxpParser.getXMLReader();
      } catch(javax.xml.parsers.ParserConfigurationException ex) {
      	throw new org.xml.sax.SAXException(ex);
      } catch(javax.xml.parsers.FactoryConfigurationError ex1) {
      	throw new org.xml.sax.SAXException(ex1.toString());
      } catch(NoSuchMethodError ex2) {
      }
      if (reader == null)
         reader = XMLReaderFactory.createXMLReader();
      // create and configure an XMLFilter to use for transformation
      XMLFilter filter
         = ((SAXTransformerFactory) tfactory).newXMLFilter(new StreamSource(xslID));
      filter.setParent(reader);
      filter.setContentHandler(new oraContentHandler());
      // transform the XML document
      filter.parse(new InputSource(xmlID));
   }


   /**
    * Performs a chain of transformations.
    * @param xmlID   - the source XML
    * @param xslID_1 - the stylesheet 1
    * @param xslID_2 - the stylesheet 2
    * @param xslID_3 - the stylesheet 3
    */ 
   public static void xmlFilterChain(
      String xmlID, String xslID_1,
      String xslID_2, String xslID_3)
      throws TransformerException, TransformerConfigurationException,
      SAXException, IOException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();
      if (tfactory.getFeature(SAXSource.FEATURE))
      {
         SAXTransformerFactory stf = (SAXTransformerFactory)tfactory;
         XMLReader reader = null;
         try {
	    javax.xml.parsers.SAXParserFactory factory =
      	       javax.xml.parsers.SAXParserFactory.newInstance();
            factory.setNamespaceAware(true);
            // instantiate an XMLReader for parsing
	    javax.xml.parsers.SAXParser jaxpParser =
	       factory.newSAXParser();
	    reader = jaxpParser.getXMLReader();
         } catch(javax.xml.parsers.ParserConfigurationException ex) {
	         throw new org.xml.sax.SAXException( ex );
         } catch(javax.xml.parsers.FactoryConfigurationError ex1) {
	         throw new org.xml.sax.SAXException( ex1.toString() );
         } catch(NoSuchMethodError ex2) {
         }
         if (reader == null ) reader = XMLReaderFactory.createXMLReader();
         // create three XMLFilters, one for each stylesheet 
         XMLFilter filter1 = stf.newXMLFilter(new StreamSource(xslID_1));
         XMLFilter filter2 = stf.newXMLFilter(new StreamSource(xslID_2));
         XMLFilter filter3 = stf.newXMLFilter(new StreamSource(xslID_3));
         if (filter1 != null && filter2 != null && filter3 != null)
         {
	    // specify the order of the transformations
            filter1.setParent(reader);
            filter2.setParent(filter1);
            filter3.setParent(filter2);
            filter3.setContentHandler(new oraContentHandler());
            // parse the input XML document
            filter3.parse(new InputSource(xmlID));
         }
      }
   }

   /** 
    * Performs the transformation with SAX.
    * The transformation result is then passed to ContentHandler.
    * @param xmlID - the source XML
    * @param xslID - the stylesheet
    */ 
   public static void contentHandler2contentHandler(String xmlID, String xslID)
      throws TransformerException,
      TransformerConfigurationException,
      SAXException, IOException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();

      if (tfactory.getFeature(SAXSource.FEATURE))
      {
         SAXTransformerFactory stfactory = ((SAXTransformerFactory) tfactory);
         TransformerHandler handler
            = stfactory.newTransformerHandler(new StreamSource(xslID));
         Result result = new SAXResult(new oraContentHandler());
         handler.setResult(result);
         XMLReader reader = null;
         try {
	    javax.xml.parsers.SAXParserFactory factory =
	       javax.xml.parsers.SAXParserFactory.newInstance();
            factory.setNamespaceAware(true);
	    javax.xml.parsers.SAXParser jaxpParser =
	       factory.newSAXParser();
            // create an XML reader for parsing
            reader=jaxpParser.getXMLReader();
         } catch( javax.xml.parsers.ParserConfigurationException ex ) {
	         throw new org.xml.sax.SAXException( ex );
         } catch( javax.xml.parsers.FactoryConfigurationError ex1 ) {
	         throw new org.xml.sax.SAXException( ex1.toString() );
         } catch( NoSuchMethodError ex2 ) {
         }
         if( reader == null ) reader = XMLReaderFactory.createXMLReader();
         // configure the reader
         reader.setContentHandler(handler);
         reader.setProperty("http://xml.org/sax/properties/lexical-handler", handler);
         InputSource inputSource = new InputSource(xmlID);
         // transform the XML document
         reader.parse(inputSource);
      }
   }

   /** 
    * Performs the transformation with SAX 
    * The transformation result is a DOMResult.
    * @param xmlID - the source XML
    * @param xslID - the stylesheet
    */
   public static void contentHandler2DOM(String xmlID, String xslID)
      throws TransformerException, TransformerConfigurationException,
      SAXException, IOException, ParserConfigurationException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();

      if (tfactory.getFeature(SAXSource.FEATURE)
         && tfactory.getFeature(DOMSource.FEATURE))
      {
         SAXTransformerFactory sfactory = (SAXTransformerFactory) tfactory;

         // obtain a DOM Document object to use for building the DOM tree
         DocumentBuilderFactory dfactory
            = DocumentBuilderFactory.newInstance();
         DocumentBuilder docBuilder = dfactory.newDocumentBuilder();
         org.w3c.dom.Document outNode = docBuilder.newDocument();

         TransformerHandler handler
            = sfactory.newTransformerHandler(new StreamSource(xslID));
         handler.setResult(new DOMResult(outNode));

         XMLReader reader = null;

         try {
	    javax.xml.parsers.SAXParserFactory factory =
	       javax.xml.parsers.SAXParserFactory.newInstance();
	    factory.setNamespaceAware(true);
	    javax.xml.parsers.SAXParser jaxpParser =
   	       factory.newSAXParser();
	    reader = jaxpParser.getXMLReader();
         } catch(javax.xml.parsers.ParserConfigurationException ex) {
	         throw new org.xml.sax.SAXException(ex);
         } catch(javax.xml.parsers.FactoryConfigurationError ex1) {
	         throw new org.xml.sax.SAXException(ex1.toString());
         } catch(NoSuchMethodError ex2) {
         }
         // create and configure the parser
         if(reader == null ) reader = XMLReaderFactory.createXMLReader();
         reader.setContentHandler(handler);
         reader.setProperty("http://xml.org/sax/properties/lexical-handler", handler);
         // parse and transform the input XML document to a DOM node output
         reader.parse(xmlID);
         // print the DOM node
         printDOMNode(outNode);
      }
   }

   /**
    * Transforms a DOMSource to a StreamResult.
    * In this case, print the DOM node to the System out device.
    */
   private static void printDOMNode(Node node)
      throws TransformerException, TransformerConfigurationException, SAXException, IOException,
      ParserConfigurationException
   {
      TransformerFactory tfactory = TransformerFactory.newInstance();
      Transformer serializer = tfactory.newTransformer();
      serializer.setOutputProperty(OutputKeys.METHOD, "xml");
      serializer.setOutputProperty(OutputKeys.INDENT, "no");
      serializer.transform(new DOMSource(node),
         new StreamResult(System.out));
   }

   /**
    * convert a file name to its URL format
    */
   private static URL createURL(String fileName)
   {
      URL url = null;
      try
      {
         url = new URL(fileName);
      }
      catch (MalformedURLException ex)
      {
         File f = new File(fileName);
         try
         {
            String path = f.getAbsolutePath();
            // This is a bunch of weird code that is required to
            // make a valid URL on the Windows platform, due
            // to inconsistencies in what getAbsolutePath returns.
            String fs = System.getProperty("file.separator");
            if (fs.length() == 1)
            {
               char sep = fs.charAt(0);
               if (sep != '/')
                  path = path.replace(sep, '/');
               if (path.charAt(0) != '/')
                  path = '/' + path;
            }
            path = "file://" + path;
            url = new URL(path);
         }
         catch (MalformedURLException e)
         {
            System.out.println("Cannot create url for: " + fileName);
            System.exit(0);
         }
      }
      return url;
   }
}
